const express = require('express');
const path = require('path');
const fs = require('fs');

const app = express();
app.use(express.static(path.join(__dirname, 'public')));

const bucket = "/home/codemake/bucket";
//const bucket = "c:/tmp/bucket";
const PORT = 4000;

// 🟢 CORS universal
app.use((req, res, next) => {
    res.setHeader("Access-Control-Allow-Origin", "*");
    res.setHeader("Access-Control-Allow-Methods", "GET,POST,OPTIONS");
    res.setHeader("Access-Control-Allow-Headers", "*");
    if (req.method === "OPTIONS") return res.status(204).end();
    next();
});

// 🔄 Sin límite de tamaño ni tiempo
app.use((req, res, next) => {
    req.setTimeout(0);
    res.setTimeout(0);
    next();
});


// === Página base ===
app.get('/', (req, res) => {
    res.sendFile(path.join(__dirname, 'public', 'index.html'));
});



/* Subida streaming por chunks */
app.post("/upload-stream-chunk", (req, res) => {
    const ambiente = req.query.ambiente;
    const subfolder = req.query.subfolder;
    const fileId = req.query.fileId; // nombre original completo, con extensión
    const chunkIndex = parseInt(req.query.chunkIndex, 10);
    const totalChunks = parseInt(req.query.totalChunks, 10);
    const extension = req.query.extension;

    // 🛡️ Validación de parámetros obligatorios
    const missingParams = [];
    if (!ambiente) missingParams.push("ambiente");
    if (!subfolder) missingParams.push("subfolder");
    if (!fileId) missingParams.push("fileId");
    if (isNaN(chunkIndex)) missingParams.push("chunkIndex");
    if (isNaN(totalChunks)) missingParams.push("totalChunks");
    if (!extension) missingParams.push("extension");

    if (missingParams.length > 0) {
        return res.status(400).json({
            ok: false,
            error: `Faltan parámetros requeridos: ${missingParams.join(", ")}`,
        });
    }

    // 🧱 Validación de extensión segura
    const safeExtensions = [".pdf", ".docx", ".xlsx", ".zip", ".rar", ".jpg", ".png", ".txt"];
    const finalExt = extension.startsWith(".") ? extension : `.${extension}`;
    if (!safeExtensions.includes(finalExt.toLowerCase())) {
        return res.status(400).json({
            ok: false,
            error: `Extensión no permitida: ${finalExt}`,
        });
    }

    // 🧼 Validación de nombre seguro
    if (fileId.includes("..") || fileId.includes("/") || fileId.includes("\\")) {
        return res.status(400).json({
            ok: false,
            error: "Nombre de archivo no permitido",
        });
    }

    const tempDir = path.join(bucket, ambiente, subfolder, `${fileId}_chunks`);
    const chunkPath = path.join(tempDir, `chunk_${chunkIndex}`);

    try {
        fs.mkdirSync(tempDir, { recursive: true });
    } catch (err) {
        console.error("❌ Error creando carpeta temporal:", err);
        return res.status(500).json({ ok: false, error: "Error creando carpeta temporal", detail: err.message });
    }

    const writeStream = fs.createWriteStream(chunkPath);
    req.pipe(writeStream);

    req.on("end", async () => {
        let stats;
        try {
            stats = fs.statSync(chunkPath);
        } catch (err) {
            console.error("❌ Error leyendo chunk:", err);
            return res.status(500).json({ ok: false, error: "Error leyendo chunk", detail: err.message });
        }

        console.log(`📥 Chunk ${chunkIndex + 1}/${totalChunks} recibido (${(stats.size / 1024 / 1024).toFixed(2)} MB)`);

        let filesNow;
        try {
            filesNow = fs.readdirSync(tempDir).filter(f => f.startsWith("chunk_"));
        } catch (err) {
            console.error("❌ Error leyendo carpeta de chunks:", err);
            return res.status(500).json({ ok: false, error: "Error leyendo carpeta de chunks", detail: err.message });
        }

        if (filesNow.length < totalChunks) {
            return res.json({
                ok: true,
                message: `Chunk ${chunkIndex + 1}/${totalChunks} almacenado`,
            });
        }

        console.log("🧩 Todos los chunks recibidos, combinando...");
        const finalPath = path.join(bucket, ambiente, subfolder, fileId); // nombre original intacto
        const writeFinal = fs.createWriteStream(finalPath);

        try {
            for (let i = 0; i < totalChunks; i++) {
                const chunkFile = path.join(tempDir, `chunk_${i}`);
                if (!fs.existsSync(chunkFile)) {
                    return res.status(400).json({ ok: false, error: `Falta chunk ${i}` });
                }

                await new Promise((resolve, reject) => {
                    const rs = fs.createReadStream(chunkFile);
                    rs.on("error", reject);
                    rs.on("end", () => {
                        try {
                            fs.unlinkSync(chunkFile);
                            resolve();
                        } catch (err) {
                            reject(err);
                        }
                    });
                    rs.pipe(writeFinal, { end: false });
                });
            }

            writeFinal.end();
            writeFinal.on("finish", () => {
                setTimeout(() => {
                    try {
                        fs.rmSync(tempDir, { recursive: true, force: true });
                        console.log(`🧹 Carpeta temporal eliminada: ${tempDir}`);
                    } catch (err) {
                        console.error("❌ Error eliminando carpeta temporal:", err);
                    }

                    try {
                        const finalStats = fs.statSync(finalPath);
                        console.log(`✅ Ensamblado completo → ${(finalStats.size / 1024 / 1024).toFixed(2)} MB`);
                        res.json({
                            ok: true,
                            message: "Archivo fusionado correctamente",
                            file: `${ambiente}/${subfolder}/${fileId}`,
                            sizeMB: (finalStats.size / 1024 / 1024).toFixed(2),
                        });
                    } catch (err) {
                        console.error("❌ Error leyendo archivo final:", err);
                        res.status(500).json({ ok: false, error: "Error leyendo archivo final", detail: err.message });
                    }
                }, 100); // Espera para evitar EPERM en Windows
            });
        } catch (err) {
            console.error("❌ Error combinando chunks:", err);
            res.status(500).json({ ok: false, error: "Error combinando chunks", detail: err.message });
        }
    });

    req.on("error", (err) => {
        console.error("❌ Error en transmisión:", err);
        res.status(500).json({
            ok: false,
            error: "Error de transmisión",
            detail: err.message,
        });
    });
});


app.listen(PORT, () => console.log(`🚀 Proxy corriendo en http://localhost:${PORT}`));